package bpm.gateway.core.tools;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.sql.Types;
import java.util.HashMap;

import org.json.JSONArray;
import org.json.JSONObject;

import bpm.gateway.core.StreamElement;
import bpm.vanilla.platform.core.beans.meta.Meta;
import bpm.vanilla.platform.core.beans.meta.Meta.TypeMeta;

/**
 * Mthode d'aide pour la gestion de la golocalisation des adresses
 * 
 * On prend le premier rsultat au dessus de $DEFAULT_SCORE
 */
public class AdresseGeoLocHelper {

	public static final String SCORE_EXCEPTION = "Score is lower than the minimum score.";
	private static final double DEFAULT_SCORE = 0.7;

	private static final String GOUV_ADRESSE_API = "https://api-adresse.data.gouv.fr/search/";
	private static final String PARAM_QUERY = "q";
	private static final String PARAM_POST_CODE = "postcode";
	// Not use because we don't have this code
	// private static final String PARAM_CITY_CODE = "citycode";

	public static final APIBanField GEOLOC = new APIBanField("geoloc", "Coordinate", -1, "String", Coordinate.class.getName());
	
	public static final APIBanField ID = new APIBanField("id", "identifiant de ladresse (clef dinteroprabilit)", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField TYPE = new APIBanField("type", "type de rsultat trouv", java.sql.Types.VARCHAR, "String", String.class.getName());
//	public static final APIBanField HOUSENUMBER = new APIBanField("housenumber", "numro   la plaque ", java.sql.Types.VARCHAR, "String", String.class.getName());
//	public static final APIBanField STREET = new APIBanField("street", "position   la voie , plac approximativement au centre de celle-ci", java.sql.Types.VARCHAR, "String", String.class.getName());
//	public static final APIBanField LOCALITY = new APIBanField("locality", "lieu-dit", java.sql.Types.VARCHAR, "String", String.class.getName());
//	public static final APIBanField MUNICIPALITY = new APIBanField("municipality", "numro   la commune ", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField SCORE = new APIBanField("score", "valeur de 0  1 indiquant la pertinence du rsultat", java.sql.Types.FLOAT, "Float", Double.class.getName());
	public static final APIBanField HOUSENUMBER = new APIBanField("housenumber", "numro avec indice de rptition ventuel (bis, ter, A, B)", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField STREET = new APIBanField("street", "nom de la voie", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField NAME = new APIBanField("name", "numro ventuel et nom de voie ou lieu dit", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField POSTCODE = new APIBanField("postcode", "code postal", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField CITYCODE = new APIBanField("citycode", "code INSEE de la commune", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField CITY = new APIBanField("city", "nom de la commune", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField DISTRICT = new APIBanField("district", "nom de larrondissement (Paris/Lyon/Marseille)", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField OLDCITYCODE = new APIBanField("oldcitycode", "code INSEE de la commune ancienne (le cas chant)", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField OLDCITY = new APIBanField("oldcity", "nom de la commune ancienne (le cas chant)", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField CONTEXT = new APIBanField("context", "n de dpartement, nom de dpartement et de rgion", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField LABEL = new APIBanField("label", "libell complet de ladresse", java.sql.Types.VARCHAR, "String", String.class.getName());
	public static final APIBanField X = new APIBanField("x", "coordonnes gographique en projection lgale", java.sql.Types.FLOAT, "Float", Double.class.getName());
	public static final APIBanField Y = new APIBanField("y", "coordonnes gographique en projection lgale", java.sql.Types.FLOAT, "Float", Double.class.getName());
	public static final APIBanField IMPORTANCE = new APIBanField("importance", "indicateur dimportance (champ technique)", java.sql.Types.FLOAT, "Float", Double.class.getName());
	
	public static final APIBanField[] API_FIELDS = { ID, TYPE, SCORE, HOUSENUMBER, STREET, NAME, POSTCODE, CITYCODE, CITY, DISTRICT, OLDCITYCODE, OLDCITY, CONTEXT, LABEL, X, Y, IMPORTANCE };
	
	private static AdresseGeoLocHelper instance;
	
	public static AdresseGeoLocHelper getInstance() {
		if (instance == null) {
			instance = new AdresseGeoLocHelper();
		}
		return instance;
	}

	/**
	 * Rcupration de la golocalisation d'une adresse
	 * @param onlyOneColumn 
	 * 
	 * @param logger
	 * @param adresse
	 * @return coordonnes
	 * @throws Exception 
	 */
	public static HashMap<String, Object> getGeoloc(boolean onlyOneColumn, String libelle, String codePostal, Double minimumScore) throws Exception {
		if (libelle == null || libelle.isEmpty() || codePostal == null || codePostal.isEmpty()) {
			return null;
		}
		
		StringBuffer buf = new StringBuffer();
		buf.append("?");
		buf.append(PARAM_QUERY);
		buf.append("=");
		buf.append(URLEncoder.encode(libelle, "UTF-8"));
		if (!onlyOneColumn) {
			buf.append("&");
			buf.append(PARAM_POST_CODE);
			buf.append("=");
			buf.append(URLEncoder.encode(codePostal, "UTF-8"));
		}

		String result = callUrl(GOUV_ADRESSE_API + buf.toString());
		return parseJson(result, minimumScore);
	}

	private static HashMap<String, Object> parseJson(String result, Double minimumScore) throws Exception {
		minimumScore = minimumScore != null ? minimumScore : DEFAULT_SCORE;
		
		HashMap<String, Object> values = new HashMap<String, Object>();
		
		JSONObject json = new JSONObject(result);
		if (!json.isNull("features")) {
			JSONArray features = json.getJSONArray("features");
			JSONObject item = features.length() > 0 ? features.getJSONObject(0) : null;
			if (item != null && !item.isNull("properties") && !item.isNull("geometry")) {
				JSONObject properties = item.getJSONObject("properties");
				if (!properties.isNull("score")) {
					double score = properties.getDouble("score");
					if (score > minimumScore) {
						for (APIBanField field : API_FIELDS) {
							if (!properties.isNull(field.getId())) {
								if (field.getType() == java.sql.Types.FLOAT) {
									values.put(field.getId(), properties.getDouble(field.getId()));
								}
								else {
									values.put(field.getId(), properties.getString(field.getId()));
								}
							}
						}
						
						JSONObject geometry = item.getJSONObject("geometry");
						JSONArray coordinates = geometry.getJSONArray("coordinates");
						if (coordinates.length() > 1) {
							double longitude = coordinates.getDouble(0);
							double latitude = coordinates.getDouble(1);
							
							values.put(GEOLOC.getId(), new Coordinate(latitude, longitude, score));
						}
					}
					else {
						throw new Exception(SCORE_EXCEPTION);
					}
				}
			}
		}
		
		return values;
	}

	private static String callUrl(String requestUrl) throws Exception {
		try (InputStream is = new URL(requestUrl).openStream()) {
			BufferedReader rd = new BufferedReader(new InputStreamReader(is, Charset.forName("UTF-8")));
			return readAll(rd);
		} catch (Exception e) {
			e.printStackTrace();
			throw e;
		}
	}

	private static String readAll(Reader rd) throws IOException {
		StringBuilder sb = new StringBuilder();
		int cp;
		while ((cp = rd.read()) != -1) {
			sb.append((char) cp);
		}
		return sb.toString();
	}

	public static APIBanField getField(String item) {
		for (APIBanField field : API_FIELDS) {
			if (field.getId().equals(item)) {
				return field;
			}
		}
		return null;
	}
	
	public static StreamElement buildColumn(String transfoName, APIBanField field) {
		StreamElement element = new StreamElement();
		element.className = field.getClassName();
		element.name = field.getId();
		element.tableName = "apiban";
		element.transfoName = transfoName;
		element.type = field.getType();
		element.originTransfo = transfoName;
		element.isNullable = true;
		element.defaultValue = null;
		element.typeName = field.getTypeName();
		element.isPrimaryKey = false;
		return element;
	}
}
