package bpm.gateway.runtime2.transformation.veolia;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.URL;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;
import java.util.zip.ZipOutputStream;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import bpm.gateway.core.Transformation;
import bpm.gateway.core.server.database.DataBaseConnection;
import bpm.gateway.core.server.database.DataBaseServer;
import bpm.gateway.core.server.database.jdbc.JdbcConnectionProvider;
import bpm.gateway.core.server.file.AbstractFileServer;
import bpm.gateway.core.server.file.MdmFileServer;
import bpm.gateway.core.tools.ZipHelper;
import bpm.gateway.core.veolia.ConnectorPatrimoineXML;
import bpm.gateway.core.veolia.ConnectorXML;
import bpm.gateway.core.veolia.DateChargement;
import bpm.gateway.core.veolia.LogXML;
import bpm.gateway.core.veolia.ReflectionHelper;
import bpm.gateway.core.veolia.db.VeoliaConnectionManager;
import bpm.gateway.core.veolia.db.VeoliaPatrimoineDaoComponent;
import bpm.gateway.core.veolia.patrimoine.HasPJ;
import bpm.gateway.core.veolia.patrimoine.Patrimoine;
import bpm.gateway.core.veolia.patrimoine.PatrimoineDAO;
import bpm.gateway.core.veolia.patrimoine.xls.PatrimoineXls;
import bpm.gateway.core.veolia.patrimoine.xls.PatrimoineXlsHelper;
import bpm.gateway.core.veolia.patrimoine.xls.XlsRuntimeLog;
import bpm.gateway.runtime2.RuntimeStep;
import bpm.studio.jdbc.management.model.DriverInfo;
import bpm.vanilla.platform.core.IRepositoryContext;
import bpm.vanilla.platform.core.IVanillaAPI;
import bpm.vanilla.platform.core.beans.resources.ClassDefinition;
import bpm.vanilla.platform.core.beans.resources.ClassRule;
import bpm.vanilla.platform.core.remote.RemoteVanillaPlatform;

public class RunConnectorPatrimoine extends RuntimeStep {

	private IVanillaAPI vanillaApi;

	private VeoliaPatrimoineDaoComponent veoliaDaoComp;
	private Patrimoine patrimoine;
	private PatrimoineXls patrimoineXls;
	private ClassDefinition classDef;

	private PrintWriter writer;
	private Marshaller jaxbMarshaller;

	private String beginDate, endDate;
	private String query;
	private boolean isInput;
	private boolean fullExport;
	private boolean isFromXLS;

	private String urlPJs;
	private String folderPJs;

	private DateChargement dateChargement;
	private String fileName = "";

	private List<LogXML> patrimoineLogs;

	public RunConnectorPatrimoine(IRepositoryContext repositoryContext, Transformation transformation, int bufferSize) throws Exception {
		super(repositoryContext, transformation, bufferSize);
		if (getRepositoryContext() == null) {
			throw new Exception("Cannot use a ConnectorPatrimoine step without a VanillaContext. You must be connected to Vanilla.");
		}
	}

	@Override
	public void init(Object adapter) throws Exception {
		ConnectorPatrimoineXML transfo = (ConnectorPatrimoineXML) getTransformation();
		this.vanillaApi = new RemoteVanillaPlatform(getRepositoryContext().getVanillaContext());
		this.isInput = transfo.isInput();

		if (isInput) {
			initInput(adapter, transfo);
		}
		else {
			initOutput(adapter, transfo);
		}
	}

	private void initInput(Object adapter, ConnectorPatrimoineXML transfo) throws Exception {
		JAXBContext jaxbContext = null;
		try {
			jaxbContext = JAXBContext.newInstance(Patrimoine.class);
		} catch (JAXBException e1) {
			e1.printStackTrace();
		}

		Unmarshaller jaxbUnmarshaller = null;
		try {
			jaxbUnmarshaller = jaxbContext.createUnmarshaller();
		} catch (JAXBException e) {
			e.printStackTrace();
		}

		String etlName = transfo.getDocument().getName();
		if (transfo.useMdm() && transfo.getSelectedContract() != null) {
			fileName = ((MdmFileServer) transfo.getFileServer()).getDocumentName(transfo);
		}
		else {
			fileName = transfo.getDefinition().substring(transfo.getDefinition().replace("\\", "/").lastIndexOf("/") + 1, transfo.getDefinition().length());
		}

		String format = null;
		if (transfo.useMdm() && transfo.getSelectedContract() != null) {
			format = ((MdmFileServer) transfo.getFileServer()).getFormat(Integer.parseInt(transfo.getDefinition()));
		}
		else {
			format = getFormat(fileName);
		}

		if (format != null && format.equalsIgnoreCase("xls")) {
			isFromXLS = true;
		}

		String mainClassIdentifiant = null;
		try (InputStream fis = ((AbstractFileServer) transfo.getFileServer()).getInpuStream(transfo)) {
			if (isFromXLS) {
				patrimoineLogs = new ArrayList<>();

				List<XlsRuntimeLog> logsRuntime = new ArrayList<>();
				patrimoineXls = PatrimoineXlsHelper.buildPatrimoineXls(logsRuntime, patrimoineLogs, fis, fileName);

				if (logsRuntime != null && !logsRuntime.isEmpty()) {
					for (XlsRuntimeLog log : logsRuntime) {
						if (log.isError()) {
							error(log.getMessage());
						}
						else {
							info(log.getMessage());
						}
					}
				}

				mainClassIdentifiant = patrimoineXls.getClass().getName();
			}
			else {
				patrimoine = (Patrimoine) jaxbUnmarshaller.unmarshal(fis);

				mainClassIdentifiant = patrimoine.getClass().getName();
			}

			DataBaseServer dbServer = transfo.getServer() != null ? (DataBaseServer) transfo.getServer() : null;
			if (dbServer == null) {
				throw new Exception("Please, define database server before running.");
			}

			DataBaseConnection c = (DataBaseConnection) dbServer.getCurrentConnection(adapter);
			DriverInfo driverInfo = JdbcConnectionProvider.getDriver(c.getDriverName());

			String fullUrl = "";
			if (c.isUseFullUrl()) {
				fullUrl = c.getFullUrl();
			}
			else {
				fullUrl = JdbcConnectionProvider.getFullUrl(c.getDriverName(), c.getHost(), c.getPort(), c.getDataBaseName());
			}

			if (fullUrl != null && !fullUrl.isEmpty()) {
				VeoliaConnectionManager manager = VeoliaConnectionManager.getInstance();
				veoliaDaoComp = manager.getPatrimoineDao(fullUrl, c.getLogin(), c.getPassword(), driverInfo.getClassName());
			}
			else {
				throw new Exception("Unable to get Full URL");
			}
		}

		List<ClassRule> classRules = vanillaApi.getResourceManager().getClassRules(mainClassIdentifiant);

		this.classDef = ReflectionHelper.loadClass(mainClassIdentifiant);
		ReflectionHelper.buildClassDefinitionWithRules(vanillaApi, classDef, classRules, true);

		this.dateChargement = new DateChargement(etlName, fileName, "ve_ods_patrimoine");
	}

	private void initOutput(Object adapter, ConnectorPatrimoineXML transfo) throws Exception {
		JAXBContext jaxbContext = null;
		try {
			jaxbContext = JAXBContext.newInstance(Patrimoine.class);
		} catch (JAXBException e1) {
			e1.printStackTrace();
		}

		try {
			jaxbMarshaller = jaxbContext.createMarshaller();
			jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
		} catch (JAXBException e) {
			e.printStackTrace();
		}

		this.beginDate = transfo.getBeginDate();
		this.endDate = transfo.getEndDate();
		this.query = transfo.getQuery();
		this.fullExport = transfo.isFullExport();
		
		this.urlPJs = transfo.getUrlPJs();
		this.folderPJs = transfo.getFolderPJs();

		DataBaseServer dbServer = transfo.getServer() != null ? (DataBaseServer) transfo.getServer() : null;
		if (dbServer == null) {
			throw new Exception("Please, define database server before running.");
		}

		DataBaseConnection c = (DataBaseConnection) dbServer.getCurrentConnection(adapter);
		DriverInfo driverInfo = JdbcConnectionProvider.getDriver(c.getDriverName());

		String fullUrl = "";
		if (c.isUseFullUrl()) {
			fullUrl = c.getFullUrl();
		}
		else {
			fullUrl = JdbcConnectionProvider.getFullUrl(c.getDriverName(), c.getHost(), c.getPort(), c.getDataBaseName());
		}

		if (fullUrl != null && !fullUrl.isEmpty()) {
			VeoliaConnectionManager manager = VeoliaConnectionManager.getInstance();
			veoliaDaoComp = manager.getPatrimoineDao(fullUrl, c.getLogin(), c.getPassword(), driverInfo.getClassName());
		}
		else {
			throw new Exception("Unable to get Full URL");
		}

		String fileName = null;
		try {
			fileName = transfo.getDocument().getStringParser().getValue(getTransformation().getDocument(), transfo.getDefinition());
		} catch (Exception e) {
			error(" error when getting/parsing fileName", e);
			throw e;
		}

		File f = new File(fileName);
		if (transfo.isDelete() && f.exists()) {
			f.delete();
			info(" delete file " + f.getAbsolutePath());
		}

		// flag to decide if the Headers Should be Writed or not
		boolean fileCreated = false;
		f = new File(fileName);
		if (!f.exists()) {
			try {
				f.createNewFile();
				fileCreated = true;
				info(" file " + f.getAbsolutePath() + " created");
			} catch (Exception e) {
				error(" cannot create file " + f.getName(), e);
				throw e;
			}
		}

		try {
			writer = new PrintWriter(f, transfo.getEncoding());
			info(" Writer created");
		} catch (Exception e) {
			error(" cannot create writer", e);
		}
	}

	@Override
	public void performRow() throws Exception {
		if (isInput) {
			performRowInput();
		}
		else {
			performRowOutput();
		}

		setEnd();
	}

	private void performRowInput() throws Exception {
		try {
			if (veoliaDaoComp != null) {
				PatrimoineDAO patrimoineDao = veoliaDaoComp.getPatrimoineDao();
				if (isFromXLS) {
					save(patrimoineDao, patrimoineXls);
				}
				else {
					save(patrimoineDao, patrimoine);
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			setEnd();
			throw new Exception("Unable to load the file : " + e.getMessage());
		}
	}

	private void performRowOutput() throws JAXBException {
		Patrimoine patrimoine = veoliaDaoComp.getPatrimoineDao().getPatrimoine(beginDate, endDate, query, fullExport);

		jaxbMarshaller.marshal(patrimoine, writer);
		
		try {
			if (urlPJs != null && !urlPJs.isEmpty() && folderPJs != null && !folderPJs.isEmpty()) {
				managePJs(patrimoine, urlPJs, folderPJs, query);
			}
		} catch(Exception e) {
			e.printStackTrace();
		}
	}

	private void save(PatrimoineDAO patrimoineDao, Patrimoine patrimoine) {
		patrimoineDao.save(dateChargement, patrimoine, classDef, fileName);
	}

	private void save(PatrimoineDAO patrimoineDao, PatrimoineXls patrimoineXls) {
		patrimoineDao.saveXls(dateChargement, patrimoineXls, classDef, fileName, patrimoineLogs);
	}
	
	public void managePJs(Patrimoine patrimoine, String urlPJs, String folderPJs, String contrat) {Date date = new Date();
		Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("Europe/Paris"));
		cal.setTime(date);
		int year = cal.get(Calendar.YEAR);
		int month = cal.get(Calendar.MONTH) + 1;
		int day = cal.get(Calendar.DAY_OF_MONTH);
		int hour = cal.get(Calendar.HOUR);
		int min = cal.get(Calendar.MINUTE);
		folderPJs += "/" + year + month + day + hour + min + "_" + contrat + "_PJ";
		
		List<String> sitesOuvragesUnitesPJs = new ArrayList<String>();
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvrageBars() != null ? patrimoine.getOuvrageBars().getOuvrageBars() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvrageBass() != null ? patrimoine.getOuvrageBass().getOuvrageBas() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvrageCaps() != null ? patrimoine.getOuvrageCaps().getOuvrageCaps() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvragePcas() != null ? patrimoine.getOuvragePcas().getOuvragePcas() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvrageRsts() != null ? patrimoine.getOuvrageRsts().getOuvrageRsts() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvrageSocles() != null ? patrimoine.getOuvrageSocles().getOuvrageSocles() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getOuvrageUsis() != null ? patrimoine.getOuvrageUsis().getOuvrageUsis() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getSites() != null ? patrimoine.getSites().getSite() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getUniteChls() != null ? patrimoine.getUniteChls().getUniteChl() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getUnitePoms() != null ? patrimoine.getUnitePoms().getUnitePoms() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getUniteRegs() != null ? patrimoine.getUniteRegs().getUniteRegs() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getUniteSecs() != null ? patrimoine.getUniteSecs().getUniteSecs() : null));
		sitesOuvragesUnitesPJs.addAll(getPJs(patrimoine.getUniteTlgs() != null ? patrimoine.getUniteTlgs().getUniteTlgs() : null));
		
		for (String value : sitesOuvragesUnitesPJs) {
			if (value != null && !value.isEmpty() && !value.equals("NR")) {
				retrieveFile(urlPJs, ConnectorXML.PJS_SITES_OUVRAGES_UNITES, folderPJs, value);
			}
		}
		
		List<String> eqptsIntervComposPJs = new ArrayList<String>();
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getComposants() != null ? patrimoine.getComposants().getComposant() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsAspirateursBoues() != null ? patrimoine.getEquipementsAspirateursBoues().getEquipementAspirateurBoues() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsAuscultationsBarrages() != null ? patrimoine.getEquipementsAuscultationsBarrages().getEquipementAuscultationBarrage() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsAutomates() != null ? patrimoine.getEquipementsAutomates().getEquipementAutomate() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsBallons() != null ? patrimoine.getEquipementsBallons().getEquipementBallon() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsBoitesBoue() != null ? patrimoine.getEquipementsBoitesBoue().getEquipementBoiteBoue() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsCapteurs() != null ? patrimoine.getEquipementsCapteurs().getEquipementCapteur() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsCellulesPosteHT() != null ? patrimoine.getEquipementsCellulesPosteHT().getEquipementCellulePosteHT() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsCentrifugeuses() != null ? patrimoine.getEquipementsCentrifugeuses().getEquipementCentrifugeuse() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsDestructeurOzone() != null ? patrimoine.getEquipementsDestructeurOzone().getEquipementDestructeurOzone() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsDisconnecteurs() != null ? patrimoine.getEquipementsDisconnecteurs().getEquipementDisconnecteur() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsEnregistreurs() != null ? patrimoine.getEquipementsEnregistreurs().getEquipementEnregistreur() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsGrappins() != null ? patrimoine.getEquipementsGrappins().getEquipementGrappin() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsGroupesDosage() != null ? patrimoine.getEquipementsGroupesDosage().getEquipementGroupeDosage() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsHydroejecteurs() != null ? patrimoine.getEquipementsHydroejecteurs().getEquipementHydroejecteur() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsMaterielsLaboratoire() != null ? patrimoine.getEquipementsMaterielsLaboratoire().getEquipementMaterielLaboratoire() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsMembranesUF() != null ? patrimoine.getEquipementsMembranesUF().getEquipementMembraneUF() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsMotoReducteurs() != null ? patrimoine.getEquipementsMotoReducteurs().getEquipementMotoReducteur() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsOnduleurs() != null ? patrimoine.getEquipementsOnduleurs().getEquipementOnduleur() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsOuvragesEvacuationBarrage() != null ? patrimoine.getEquipementsOuvragesEvacuationBarrage().getEquipementsOuvrageEvacuationBarrage() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsPompes() != null ? patrimoine.getEquipementsPompes().getEquipementPompe() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsProtectionsIncendie() != null ? patrimoine.getEquipementsProtectionsIncendie().getEquipementProtectionIncendie() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsRetournementsTerres() != null ? patrimoine.getEquipementsRetournementsTerres().getEquipementRetournementTerres() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsSocle() != null ? patrimoine.getEquipementsSocle().getEquipementSocle() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsTransformateursHT() != null ? patrimoine.getEquipementsTransformateursHT().getEquipementTransformateurHT() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsVariateursVitesse() != null ? patrimoine.getEquipementsVariateursVitesse().getEquipementVariateurVitesse() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsVerins() != null ? patrimoine.getEquipementsVerins().getEquipementVerin() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getEquipementsVisTransfert() != null ? patrimoine.getEquipementsVisTransfert().getEquipementVisTransfert() : null));
		eqptsIntervComposPJs.addAll(getPJs(patrimoine.getInterventions() != null ? patrimoine.getInterventions().getIntervention() : null));

		for (String value : eqptsIntervComposPJs) {
			if (value != null && !value.isEmpty() && !value.equals("NR")) {
				retrieveFile(urlPJs, ConnectorXML.PJS_EQPTS_INTERV_COMPOS, folderPJs, value);
			}
		}
		
		//Zip file
        try {
			FileOutputStream fos = new FileOutputStream(folderPJs + ".zip");
			ZipOutputStream zipOut = new ZipOutputStream(fos);
			File fileToZip = new File(folderPJs);

			ZipHelper.zipFile(fileToZip, fileToZip.getName(), zipOut);
			zipOut.close();
			fos.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private void retrieveFile(String urlPJs, String paramPJ, String folderPJs, String value) {
		String fileUrl = urlPJs + paramPJ + "/" + value;
		String folderPath = folderPJs + paramPJ;
		if (!new File(folderPath).exists() && !new File(folderPath).mkdirs()) {
			error("Unable to create folder " + folderPath);
			return;
		}
		
		try (ReadableByteChannel readableByteChannel = Channels.newChannel(new URL(fileUrl).openStream()); FileOutputStream fileOutputStream = new FileOutputStream(folderPath + "/" + value)) {
			FileChannel fileChannel = fileOutputStream.getChannel();
			fileChannel.transferFrom(readableByteChannel, 0, Long.MAX_VALUE);
		} catch(Exception e) {
			e.printStackTrace();
			warn("Unable to get the PJ '" + value + "'");
		}
	}

	private List<String> getPJs(List<? extends HasPJ> hasPJs) {
		List<String> pjs = new ArrayList<String>();
		if (hasPJs != null) {
			for (HasPJ hasPJ : hasPJs) {
				pjs.addAll(hasPJ.getPJs());
			}
		}
		return pjs;
	}

	private String getFormat(String newFileName) {
		int index = newFileName.lastIndexOf(".") + 1;
		return newFileName.substring(index);
	}

	@Override
	public void releaseResources() {
		this.veoliaDaoComp = null;
		this.patrimoine = null;
		this.patrimoineXls = null;

		if (writer != null) {
			this.writer.close();
		}
		this.writer = null;
		this.jaxbMarshaller = null;

		this.isInput = false;
		this.beginDate = null;
		this.endDate = null;
		this.query = null;

		this.vanillaApi = null;
		this.classDef = null;
		this.patrimoineLogs = null;
	}
}
